#!/usr/bin/env python3
"""
Slack App Setup Helper for Automation Demo
==========================================

This script helps you set up a Slack app for the automation demo.
It provides step-by-step instructions and validates your configuration.

Usage:
    python setup_slack.py
"""

import os
import json
import webbrowser
from typing import Dict, Any

def print_header():
    """Print the setup header"""
    print("\n" + "="*60)
    print("🔧 SLACK APP SETUP HELPER")
    print("="*60)
    print("This script will help you set up Slack integration")
    print("for the Plausible Analytics automation demo.")
    print("="*60)

def check_existing_token():
    """Check if Slack token is already configured"""
    token = os.getenv('SLACK_BOT_TOKEN')
    if token:
        print("✅ Slack bot token found in environment variables")
        return True
    else:
        print("❌ No Slack bot token found")
        return False

def open_slack_api():
    """Open Slack API website"""
    print("\n🌐 Opening Slack API website...")
    webbrowser.open('https://api.slack.com/apps')
    input("Press Enter after creating your Slack app...")

def get_app_details():
    """Get app details from user"""
    print("\n📝 Please provide your Slack app details:")

    app_name = input("App Name (e.g., 'Analytics Bot'): ").strip()
    if not app_name:
        app_name = "Analytics Automation Bot"

    return {
        "name": app_name,
        "description": f"Automated notifications for {app_name}"
    }

def generate_scopes_config():
    """Generate the required OAuth scopes"""
    scopes = [
        "chat:write",
        "chat:write.public",
        "channels:read",
        "users:read"
    ]

    print("\n🔑 Required OAuth Scopes:")
    for scope in scopes:
        print(f"  • {scope}")

    return scopes

def create_env_template():
    """Create environment template"""
    template = """# Slack Configuration for Automation Demo
SLACK_BOT_TOKEN=xoxb-your-bot-token-here
SLACK_CHANNEL=#automation-demo

# Optional: Plausible Configuration for Live Mode
PLAUSIBLE_API_KEY=your-plausible-api-key
PLAUSIBLE_SITE_ID=your-site-id
"""

    with open('.env.template', 'w') as f:
        f.write(template)

    print("\n📄 Created .env.template file")
    print("Copy this to .env and fill in your actual tokens")

def test_slack_connection(token: str, channel: str = "#automation-demo"):
    """Test Slack connection"""
    try:
        from slack_sdk import WebClient
        from slack_sdk.errors import SlackApiError

        client = WebClient(token=token)

        # Test authentication
        response = client.auth_test()
        print(f"✅ Connected as: {response['user']}")

        # Test channel access
        channels = client.conversations_list()
        channel_found = False
        for ch in channels['channels']:
            if ch['name'] == channel.lstrip('#'):
                channel_found = True
                break

        if channel_found:
            print(f"✅ Channel access confirmed: {channel}")
        else:
            print(f"⚠️ Channel {channel} not found. Available channels:")
            for ch in channels['channels'][:5]:  # Show first 5
                print(f"  • #{ch['name']}")

        # Send test message
        response = client.chat_postMessage(
            channel=channel,
            text="🧪 Test message from Automation Demo setup!",
            username="Setup Bot",
            icon_emoji="🔧"
        )
        print(f"✅ Test message sent successfully!")

        return True

    except SlackApiError as e:
        print(f"❌ Slack API Error: {e}")
        return False
    except Exception as e:
        print(f"❌ Connection test failed: {e}")
        return False

def main():
    """Main setup function"""
    print_header()

    if check_existing_token():
        test_choice = input("\nTest existing connection? (y/n): ").lower().strip()
        if test_choice == 'y':
            token = os.getenv('SLACK_BOT_TOKEN')
            channel = os.getenv('SLACK_CHANNEL', '#automation-demo')
            test_slack_connection(token, channel)
        return

    print("\n🚀 Starting Slack App Setup...")
    print("\nStep 1: Create Slack App")
    open_slack_api()

    app_details = get_app_details()

    print("\nStep 2: Configure OAuth Scopes")
    generate_scopes_config()

    print("\nStep 3: Install App to Workspace")
    print("• Go to 'OAuth & Permissions' in your Slack app")
    print("• Click 'Install to Workspace'")
    print("• Authorize the app")
    print("• Copy the 'Bot User OAuth Token' (starts with xoxb-)")

    token = input("\nPaste your Bot User OAuth Token: ").strip()
    if not token:
        print("❌ No token provided. Setup cancelled.")
        return

    # Test the connection
    print("\n🔍 Testing Slack connection...")
    if test_slack_connection(token):
        print("\n✅ Setup completed successfully!")

        # Create environment file
        create_env_template()

        print("\n📝 Next steps:")
        print("1. Copy .env.template to .env")
        print("2. Add your tokens to .env")
        print("3. Run: python automation_demo.py --demo")
        print("\n🎉 You're ready to automate!")

    else:
        print("\n❌ Setup failed. Please check your configuration and try again.")

if __name__ == "__main__":
    main()