#!/usr/bin/env python3
"""
Plausible Analytics to Slack Automation Demo
===========================================

This script demonstrates real-time automation that monitors Plausible Analytics
and sends Slack notifications when key milestones are achieved.

Features:
- Real-time data fetching from Plausible API
- Milestone detection with configurable thresholds
- Slack notifications with rich formatting
- Comprehensive logging and error handling
- Demo mode with simulated data

Usage:
    python automation_demo.py --demo    # Run with simulated data
    python automation_demo.py --live    # Run with real API calls (requires API keys)
"""

import os
import json
import time
import logging
import argparse
from datetime import datetime, timedelta
from typing import Dict, List, Optional, Any
import requests
from slack_sdk import WebClient
from slack_sdk.errors import SlackApiError
import pandas as pd
import matplotlib.pyplot as plt
import seaborn as sns

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler('automation_demo.log'),
        logging.StreamHandler()
    ]
)
logger = logging.getLogger(__name__)

class PlausibleAPI:
    """Plausible Analytics API client"""

    def __init__(self, api_key: str, site_id: str):
        self.api_key = api_key
        self.site_id = site_id
        self.base_url = "https://plausible.io/api/v1"
        self.headers = {
            "Authorization": f"Bearer {api_key}",
            "Content-Type": "application/json"
        }

    def get_realtime_stats(self) -> Dict[str, Any]:
        """Fetch current visitor statistics"""
        try:
            url = f"{self.base_url}/stats/realtime/visitors"
            params = {"site_id": self.site_id}
            response = requests.get(url, headers=self.headers, params=params, timeout=10)
            response.raise_for_status()
            return response.json()
        except requests.RequestException as e:
            logger.error(f"Failed to fetch realtime stats: {e}")
            return {"visitors": 0}

    def get_aggregate_stats(self, period: str = "30d") -> Dict[str, Any]:
        """Fetch aggregate statistics for a time period"""
        try:
            url = f"{self.base_url}/stats/aggregate"
            params = {
                "site_id": self.site_id,
                "period": period,
                "metrics": "visitors,pageviews,bounce_rate,visit_duration"
            }
            response = requests.get(url, headers=self.headers, params=params, timeout=10)
            response.raise_for_status()
            return response.json()
        except requests.RequestException as e:
            logger.error(f"Failed to fetch aggregate stats: {e}")
            return {
                "visitors": {"value": 0},
                "pageviews": {"value": 0},
                "bounce_rate": {"value": 0},
                "visit_duration": {"value": 0}
            }

class SlackNotifier:
    """Slack notification handler"""

    def __init__(self, bot_token: str):
        self.client = WebClient(token=bot_token)

    def send_milestone_notification(self, milestone: Dict[str, Any], stats: Dict[str, Any]) -> bool:
        """Send formatted milestone notification to Slack"""
        try:
            message = self._format_milestone_message(milestone, stats)

            response = self.client.chat_postMessage(
                channel="#automation-demo",
                text=message['text'],
                blocks=message['blocks'],
                username="🚀 Milestone Bot",
                icon_emoji="🎯"
            )

            logger.info(f"✅ Milestone notification sent: {milestone['name']} to #{response['channel']}")
            return True

        except SlackApiError as e:
            logger.error(f"❌ Failed to send Slack notification: {e}")
            return False

    def _format_milestone_message(self, milestone: Dict[str, Any], stats: Dict[str, Any]) -> Dict[str, Any]:
        """Format milestone data into Slack message blocks"""
        current_value = stats.get(milestone['metric'], 0)
        threshold = milestone['threshold']

        # Determine milestone status
        achieved = current_value >= threshold
        emoji = "🎉" if achieved else "📈"
        status_text = "ACHIEVED!" if achieved else "In Progress"

        blocks = [
            {
                "type": "header",
                "text": {
                    "type": "plain_text",
                    "text": f"{emoji} {milestone['name']}"
                }
            },
            {
                "type": "section",
                "text": {
                    "type": "mrkdwn",
                    "text": f"*{status_text}*\n{milestone['message']}"
                }
            },
            {
                "type": "section",
                "fields": [
                    {
                        "type": "mrkdwn",
                        "text": f"*Current Value:*\n{current_value:,.0f}"
                    },
                    {
                        "type": "mrkdwn",
                        "text": f"*Target:*\n{threshold:,.0f}"
                    },
                    {
                        "type": "mrkdwn",
                        "text": f"*Progress:*\n{max(0, min(100, (current_value / threshold) * 100)):.1f}%"
                    },
                    {
                        "type": "mrkdwn",
                        "text": f"*Last Updated:*\n{datetime.now().strftime('%H:%M:%S')}"
                    }
                ]
            }
        ]

        # Add celebration for achieved milestones
        if achieved:
            blocks.append({
                "type": "section",
                "text": {
                    "type": "mrkdwn",
                    "text": "🎊 *Congratulations!* This milestone has been achieved!"
                }
            })

        return {
            "text": f"{emoji} {milestone['name']}: {status_text}",
            "blocks": blocks
        }

class MilestoneDetector:
    """Detects when milestones are achieved"""

    def __init__(self, config_file: str = "milestones.json"):
        self.config_file = config_file
        self.milestones = self._load_milestones()
        self.achieved_milestones = set()

    def _load_milestones(self) -> List[Dict[str, Any]]:
        """Load milestone configuration"""
        try:
            with open(self.config_file, 'r', encoding='utf-8') as f:
                return json.load(f)
        except FileNotFoundError:
            logger.warning(f"Config file {self.config_file} not found, using defaults")
            return self._get_default_milestones()

    def _get_default_milestones(self) -> List[Dict[str, Any]]:
        """Return default milestone configuration"""
        return [
            {
                "id": "visitors_100",
                "name": "100 Visitors Milestone",
                "metric": "visitors",
                "threshold": 100,
                "message": "We've reached 100 visitors! 🎯",
                "celebration": "First major milestone achieved!"
            },
            {
                "id": "visitors_500",
                "name": "500 Visitors Celebration",
                "metric": "visitors",
                "threshold": 500,
                "message": "Halfway to 1000 visitors! 📈",
                "celebration": "Growing steadily!"
            },
            {
                "id": "visitors_1000",
                "name": "1000 Visitors Goal",
                "metric": "visitors",
                "threshold": 1000,
                "message": "🎉 1000 visitors achieved! Major milestone!",
                "celebration": "Time to scale up!"
            }
        ]

    def check_milestones(self, stats: Dict[str, Any]) -> List[Dict[str, Any]]:
        """Check which milestones have been achieved"""
        newly_achieved = []

        for milestone in self.milestones:
            milestone_id = milestone['id']
            metric = milestone['metric']
            threshold = milestone['threshold']
            current_value = stats.get(metric, 0)

            if current_value >= threshold and milestone_id not in self.achieved_milestones:
                newly_achieved.append(milestone)
                self.achieved_milestones.add(milestone_id)
                logger.info(f"🏆 Milestone achieved: {milestone['name']} ({current_value} >= {threshold})")

        return newly_achieved

class DemoDataGenerator:
    """Generates simulated analytics data for demo purposes"""

    def __init__(self):
        self.start_time = datetime.now()
        self.base_visitors = 50
        self.growth_rate = 1.15  # 15% growth per minute

    def generate_stats(self) -> Dict[str, Any]:
        """Generate realistic demo statistics"""
        minutes_elapsed = (datetime.now() - self.start_time).total_seconds() / 60

        # Simulate exponential growth with some randomness
        visitors = int(self.base_visitors * (self.growth_rate ** minutes_elapsed))
        visitors += int(visitors * 0.1 * (0.5 - time.time() % 1))  # Add some noise

        return {
            "visitors": max(0, visitors),
            "pageviews": max(0, int(visitors * 2.3)),  # ~2.3 pageviews per visitor
            "bounce_rate": 0.42,  # 42% bounce rate
            "visit_duration": 180,  # 3 minutes average
            "timestamp": datetime.now().isoformat()
        }

class AutomationDemo:
    """Main automation demo controller"""

    def __init__(self, demo_mode: bool = True):
        self.demo_mode = demo_mode
        self.plausible = None
        self.slack = None

        if not demo_mode:
            self._setup_live_clients()

        self.detector = MilestoneDetector()
        self.notifier = SlackNotifier(os.getenv('SLACK_BOT_TOKEN', 'demo-token'))
        self.demo_generator = DemoDataGenerator() if demo_mode else None

        logger.info(f"🚀 Automation Demo initialized (Demo Mode: {demo_mode})")

    def _setup_live_clients(self):
        """Setup real API clients"""
        plausible_key = os.getenv('PLAUSIBLE_API_KEY')
        site_id = os.getenv('PLAUSIBLE_SITE_ID')

        if not plausible_key or not site_id:
            raise ValueError("PLAUSIBLE_API_KEY and PLAUSIBLE_SITE_ID required for live mode")

        self.plausible = PlausibleAPI(plausible_key, site_id)
        logger.info("✅ Live API clients configured")

    def get_current_stats(self) -> Dict[str, Any]:
        """Get current analytics statistics"""
        if self.demo_mode:
            return self.demo_generator.generate_stats()
        else:
            stats = self.plausible.get_aggregate_stats("7d")
            # Normalize the response format
            return {
                "visitors": stats.get("visitors", {}).get("value", 0),
                "pageviews": stats.get("pageviews", {}).get("value", 0),
                "bounce_rate": stats.get("bounce_rate", {}).get("value", 0),
                "visit_duration": stats.get("visit_duration", {}).get("value", 0),
                "timestamp": datetime.now().isoformat()
            }

    def run_cycle(self) -> Dict[str, Any]:
        """Run one automation cycle"""
        stats = self.get_current_stats()
        achieved_milestones = self.detector.check_milestones(stats)

        results = {
            "timestamp": stats["timestamp"],
            "stats": stats,
            "achieved_milestones": len(achieved_milestones),
            "notifications_sent": 0
        }

        # Send notifications for achieved milestones
        for milestone in achieved_milestones:
            if self.notifier.send_milestone_notification(milestone, stats):
                results["notifications_sent"] += 1

        return results

    def run_continuous_demo(self, duration_minutes: int = 10, interval_seconds: int = 30):
        """Run continuous demo for specified duration"""
        logger.info(f"🎬 Starting continuous demo for {duration_minutes} minutes")

        end_time = datetime.now() + timedelta(minutes=duration_minutes)
        cycle_count = 0

        print("\n" + "="*60)
        print("🎯 PLAUSIBLE ANALYTICS AUTOMATION DEMO")
        print("="*60)
        print(f"Duration: {duration_minutes} minutes")
        print(f"Check Interval: {interval_seconds} seconds")
        print(f"Demo Mode: {'✅ Enabled' if self.demo_mode else '❌ Disabled (Live API)'}")
        print("="*60)

        try:
            while datetime.now() < end_time:
                cycle_count += 1
                result = self.run_cycle()

                # Display current status
                print(f"\n📊 Cycle {cycle_count} - {result['timestamp']}")
                print(f"   Visitors: {result['stats']['visitors']:,}")
                print(f"   Pageviews: {result['stats']['pageviews']:,}")
                print(f"   Bounce Rate: {result['stats']['bounce_rate']:.1%}")
                print(f"   Avg Duration: {result['stats']['visit_duration']:.0f}s")

                if result['achieved_milestones'] > 0:
                    print(f"   🏆 {result['achieved_milestones']} milestone(s) achieved!")
                    print(f"   📤 {result['notifications_sent']} notification(s) sent")

                time.sleep(interval_seconds)

        except KeyboardInterrupt:
            logger.info("⏹️ Demo interrupted by user")

        print("\n" + "="*60)
        print("🎬 DEMO COMPLETED")
        print("="*60)
        print(f"Total Cycles: {cycle_count}")
        print(f"Demo Mode: {'✅ Simulated Data' if self.demo_mode else '❌ Live API'}")
        print("="*60)

def main():
    """Main entry point"""
    parser = argparse.ArgumentParser(description="Plausible Analytics Automation Demo")
    parser.add_argument('--demo', action='store_true', default=True,
                       help='Run in demo mode with simulated data (default)')
    parser.add_argument('--live', action='store_true',
                       help='Run with live Plausible API (requires API keys)')
    parser.add_argument('--duration', type=int, default=10,
                       help='Demo duration in minutes (default: 10)')
    parser.add_argument('--interval', type=int, default=30,
                       help='Check interval in seconds (default: 30)')

    args = parser.parse_args()

    # Override demo mode if live is specified
    demo_mode = not args.live

    try:
        demo = AutomationDemo(demo_mode=demo_mode)
        demo.run_continuous_demo(
            duration_minutes=args.duration,
            interval_seconds=args.interval
        )

    except Exception as e:
        logger.error(f"❌ Demo failed: {e}")
        print(f"\n❌ Error: {e}")
        print("\nFor live mode, set these environment variables:")
        print("  export PLAUSIBLE_API_KEY='your_api_key'")
        print("  export PLAUSIBLE_SITE_ID='your_site_id'")
        print("  export SLACK_BOT_TOKEN='your_slack_token'")
        return 1

    return 0

if __name__ == "__main__":
    exit(main())